<?php
/**
 * Plugin Name: CF7 Stealth Acceptance (Network-ready)
 * Description: Добавляет во все формы невидимую галочку перед [submit] для защиты от спама. Поддерживает мультисайты и одиночные сайты. Вкл/выкл и автокаталог правок.
 * Version: 1.0.0
 * Author: Nickolay Andreev
 * License: GPLv2 or later
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

class CF7_Stealth_Acceptance {

    const ACCEPT_TAG = '[acceptance agree class:agree default:on invert]';
    const OPT_SITE_ENABLED = 'cf7_sa_enabled';
    const OPT_NETWORK_FORCE = 'cf7_sa_network_force';

    public function __construct() {
        // Админ страницы
        add_action('admin_menu', [$this, 'register_site_settings_page']);
        if ( is_multisite() ) {
            add_action('network_admin_menu', [$this, 'register_network_settings_page']);
        }

        // Сохранение настроек
        add_action('admin_init', [$this, 'register_settings']);

        // Вывод CSS/JS в подвале фронта
        add_action('wp_footer', [$this, 'print_footer_assets']);

        // При активации плагина — регистрируем опции по месту (без автозапуска миграций)
        register_activation_hook(__FILE__, [$this, 'on_activate']);
        register_uninstall_hook(__FILE__, ['CF7_Stealth_Acceptance', 'on_uninstall']);
    }

    /** ======= Settings & Admin UI ======= */

    public function register_site_settings_page() {
        $parent = $this->cf7_menu_exists() ? 'wpcf7' : 'options-general.php';
        add_submenu_page(
            $parent,
            'Стелс-галочка (CF7)',
            'Стелс-галочка (CF7)',
            'manage_options',
            'cf7-sa-settings',
            [$this, 'render_site_settings_page']
        );
    }

    public function register_network_settings_page() {
        add_submenu_page(
            'settings.php',
            'Стелс-галочка (CF7) — сеть',
            'Стелс-галочка (CF7)',
            'manage_network_options',
            'cf7-sa-network-settings',
            [$this, 'render_network_settings_page']
        );
    }

    public function register_settings() {
        // Опция сайта
        register_setting('cf7_sa_settings_group', self::OPT_SITE_ENABLED, [
            'type' => 'boolean',
            'sanitize_callback' => function($v){ return (bool)$v; },
            'default' => false,
        ]);

        // Опция сети
        if ( is_multisite() && is_network_admin() ) {
            register_setting('cf7_sa_network_settings_group', self::OPT_NETWORK_FORCE, [
                'type' => 'boolean',
                'sanitize_callback' => function($v){ return (bool)$v; },
                'default' => false,
            ]);
        }

        // Обработка POST — применяем массовые изменения при переключении
        if ( is_admin() && isset($_POST['_cf7_sa_action']) && check_admin_referer('cf7_sa_apply_changes') ) {
            $action = sanitize_text_field($_POST['_cf7_sa_action']);

            if ( 'save_site' === $action && current_user_can('manage_options') ) {
                $enabled = ! empty($_POST[self::OPT_SITE_ENABLED]);
                update_option(self::OPT_SITE_ENABLED, $enabled);

                // Применяем ко всем формам текущего сайта
                $this->apply_to_current_site($enabled);

                add_action('admin_notices', function() use ($enabled){
                    echo '<div class="updated"><p>Стелс-галочка ' . ($enabled ? 'включена' : 'выключена') . ' для текущего сайта. Формы обновлены.</p></div>';
                });
            }

            if ( is_multisite() && 'save_network' === $action && current_user_can('manage_network_options') ) {
                $force = ! empty($_POST[self::OPT_NETWORK_FORCE]);
                update_site_option(self::OPT_NETWORK_FORCE, $force);

                // Применяем ко всем сайтам сети
                $this->apply_to_network($force);

                add_action('network_admin_notices', function() use ($force){
                    echo '<div class="updated"><p>Стелс-галочка ' . ($force ? 'принудительно включена' : 'отключена') . ' на всех сайтах сети. Формы обновлены.</p></div>';
                });
            }
        }
    }

    public function render_site_settings_page() {
        if ( ! current_user_can('manage_options') ) { wp_die('Недостаточно прав.'); }

        $enabled = $this->is_effectively_enabled_for_current_site();
        $site_val = (bool) get_option(self::OPT_SITE_ENABLED, false);
        $network_forced = $this->is_network_forced();

        ?>
        <div class="wrap">
            <h1>Стелс-галочка для форм (CF7)</h1>
            <?php if ( $network_forced ): ?>
                <div class="notice notice-info"><p>Внимание: в сети активирован режим «Принудительно включить на всех сайтах». Этот сайт будет включён независимо от локальной настройки.</p></div>
            <?php endif; ?>
            <form method="post" action="">
                <?php wp_nonce_field('cf7_sa_apply_changes'); ?>
                <input type="hidden" name="_cf7_sa_action" value="save_site">
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Плагин активен</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPT_SITE_ENABLED); ?>" value="1" <?php checked($site_val, true); ?> <?php disabled($network_forced); ?>>
                                Включить стелс-галочку на всех формах сайта
                            </label>
                            <p class="description">Если включено, в шаблоны форм автоматически добавляется скрытая галочка перед кнопкой отправки; в подвале сайта выводятся CSS и JS. При выключении — галочка из форм удаляется, CSS/JS перестают выводиться.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Сохранить и применить'); ?>
            </form>

            <h2>Статус</h2>
            <p>Фактическое состояние для сайта: <strong><?php echo $enabled ? 'ВКЛ' : 'ВЫКЛ'; ?></strong></p>
        </div>
        <?php
    }

    public function render_network_settings_page() {
        if ( ! current_user_can('manage_network_options') ) { wp_die('Недостаточно прав.'); }
        $force = $this->is_network_forced();
        ?>
        <div class="wrap">
            <h1>Стелс-галочка (CF7) — настройки сети</h1>
            <form method="post" action="">
                <?php wp_nonce_field('cf7_sa_apply_changes'); ?>
                <input type="hidden" name="_cf7_sa_action" value="save_network">
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Принудительно включить на всех сайтах</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPT_NETWORK_FORCE); ?>" value="1" <?php checked($force, true); ?>>
                                Включить на всех сайтах мультисети, игнорируя локальные настройки
                            </label>
                            <p class="description">При включении — во всех формах на всех сайтах сети появится скрытая галочка, а CSS/JS будут выводиться в подвале. При выключении — изменения будут откатаны во всей сети.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Сохранить и применить по сети'); ?>
            </form>
        </div>
        <?php
    }

    /** ======= Apply / Revert ======= */

    /** Применить к текущему сайту (вставка или удаление галочки во всех формах) */
    private function apply_to_current_site( bool $enable ) {
        if ( ! $this->cf7_exists() ) return;

        $forms = get_posts([
            'post_type'      => 'wpcf7_contact_form',
            'post_status'    => 'any',
            'posts_per_page' => -1,
            'orderby'        => 'ID',
            'order'          => 'ASC',
        ]);

        foreach ( $forms as $form_post ) {
            $obj = \WPCF7_ContactForm::get_instance( (int)$form_post->ID );
            if ( ! $obj ) { continue; }

            $props = $obj->get_properties();
            $tpl = isset($props['form']) ? (string)$props['form'] : '';

            if ( '' === trim($tpl) ) { continue; }

            if ( $enable ) {
                // вставить если нет
                if ( ! $this->has_acceptance_agree($tpl) && $this->has_submit($tpl) ) {
                    $new = $this->insert_acceptance_before_submit($tpl);
                    if ( $new !== $tpl ) {
                        $props['form'] = $new;
                        $obj->set_properties($props);
                        $obj->save();
                        update_post_meta($form_post->ID, 'form', $new);
                    }
                }
            } else {
                // удалить, если мы вставляли
                if ( $this->has_acceptance_agree($tpl) ) {
                    $new = $this->remove_our_acceptance($tpl);
                    if ( $new !== $tpl ) {
                        $props['form'] = $new;
                        $obj->set_properties($props);
                        $obj->save();
                        update_post_meta($form_post->ID, 'form', $new);
                    }
                }
            }
        }
    }

    /** Применить ко всей сети */
    private function apply_to_network( bool $enable ) {
        if ( ! is_multisite() ) return;

        $sites = get_sites(['number' => 0]);
        foreach ( $sites as $site ) {
            switch_to_blog( (int) $site->blog_id );
            $this->apply_to_current_site($enable);
            restore_current_blog();
        }
    }

    /** ======= Front assets (footer) ======= */

    public function print_footer_assets() {
        if ( ! $this->is_effectively_enabled_for_current_site() ) return;

        // CSS: скрыть чекбокс
        echo "<style>.agree{display:none !important;}</style>\n";

        // JS: снять галочку (jQuery безопасно: грузим только если есть)
        // Если jQuery отсутствует, сделаем нативный дубль.
        ?>
<script>
(function(){
  var run = function(){
    // jQuery
    if (window.jQuery) {
      jQuery('.agree').prop('checked', false);
    }
    // Native fallback
    var x = document.getElementsByClassName('agree');
    for (var i=0;i<x.length;i++){ x[i].checked = false; }
  };
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', run);
  } else {
    run();
  }
  // на случай динамических форм (AJAX и CF7 события)
  document.addEventListener('wpcf7invalid', run, false);
  document.addEventListener('wpcf7mailsent', run, false);
})();
</script>
        <?php
    }

    /** ======= Helpers ======= */

    private function cf7_menu_exists(): bool {
        global $menu;
        // У CF7 верхний slug — wpcf7
        return class_exists('WPCF7') || has_action('admin_menu', 'wpcf7_admin_menu');
    }

    private function cf7_exists(): bool {
        return class_exists('\WPCF7_ContactForm');
    }

    private function is_network_forced(): bool {
        return is_multisite() ? (bool) get_site_option(self::OPT_NETWORK_FORCE, false) : false;
    }

    private function is_effectively_enabled_for_current_site(): bool {
        // если в сети включено принудительно — считаем включённым
        if ( $this->is_network_forced() ) return true;
        return (bool) get_option(self::OPT_SITE_ENABLED, false);
    }

    private function has_acceptance_agree( string $tpl ): bool {
        return (bool) preg_match('/\[acceptance\b(?=[^\]]*class\s*:\s*agree)[^\]]*\]/i', $tpl);
    }

    private function has_submit( string $tpl ): bool {
        return (bool) preg_match('/\[submit\b[^\]]*\]/i', $tpl);
    }

    private function insert_acceptance_before_submit( string $tpl ): string {
        $accept = self::ACCEPT_TAG;
        $result = preg_replace('/\[submit\b[^\]]*\]/i', $accept . "\n" . '$0', $tpl, 1);
        return is_string($result) ? $result : $tpl;
    }

    private function remove_our_acceptance( string $tpl ): string {
        // Удаляем ровно нашу строку (возможны пробелы/переносы до/после)
        $pattern = '/\s*\[acceptance\s+agree\s+(?=[^\]]*class\s*:\s*agree)(?=[^\]]*default\s*:\s*on)(?=[^\]]*invert)[^\]]*\]\s*\n?/i';
        $new = preg_replace($pattern, '', $tpl, 1);
        return is_string($new) ? $new : $tpl;
    }

    /** Активатор/деинсталлятор */

    public function on_activate() {
        // Ничего «автоматически» не включаем; пользователь сам решает на странице настроек.
        if ( is_multisite() ) {
            // Создадим опцию сети по умолчанию
            if ( false === get_site_option(self::OPT_NETWORK_FORCE, null) ) {
                add_site_option(self::OPT_NETWORK_FORCE, false);
            }
        }
        if ( false === get_option(self::OPT_SITE_ENABLED, null) ) {
            add_option(self::OPT_SITE_ENABLED, false);
        }
    }

    public static function on_uninstall() {
        // Удаляем только наши флаги (не трогаем формы, чтобы не рисковать контентом при случайной деинсталляции).
        if ( is_multisite() ) {
            delete_site_option(self::OPT_NETWORK_FORCE);
            // По желанию можно пройтись по сети и откатить формы — это опаснее, поэтому оставим пользователю контроль.
        }
        // Сайтовые опции:
        if ( ! is_multisite() ) {
            delete_option(self::OPT_SITE_ENABLED);
        } else {
            // В мультисети чистим на всех блогах
            $sites = get_sites(['number'=>0]);
            foreach ( $sites as $site ) {
                switch_to_blog( (int) $site->blog_id );
                delete_option(self::OPT_SITE_ENABLED);
                restore_current_blog();
            }
        }
    }
}

new CF7_Stealth_Acceptance();
