<?php
/**
 * Plugin Name: Comment Stealth Timer (Network-ready)
 * Description: Защита комментариев без капчи: скрытое поле с таймером + проверка времени чтения. Идея решения взята с сайта reset.name.
 * Version: 1.0.0
 * Author: Nickolay Andreev
 * License: GPLv2 or later
 */

if ( ! defined('ABSPATH') ) exit;

class Comment_Stealth_Timer {

    // опции
    const OPT_SITE_ENABLED     = 'cst_enabled';
    const OPT_SITE_SECONDS     = 'cst_seconds';
    const OPT_SITE_FIELD       = 'cst_field';
    const OPT_SITE_STATUS_CODE = 'cst_status_code';
    const OPT_SITE_MESSAGE     = 'cst_message';
    const OPT_SITE_BYPASS_MODS = 'cst_bypass_mods';
    const OPT_SITE_HTML_MODE   = 'cst_force_xhtml';

    const OPT_NET_FORCE        = 'cst_network_force';

    // значения по умолчанию
    const DEFAULT_SECONDS      = 30;
    const DEFAULT_FIELD        = 'csrf';
    const DEFAULT_STATUS_CODE  = 503;
    const DEFAULT_MESSAGE      = 'Service Unavailable';
    
    public function __construct() {
        // админ-страницы
        add_action('admin_menu', [$this,'site_menu']);
        if ( is_multisite() ) {
            add_action('network_admin_menu', [$this,'network_menu']);
        }

        // регистрация настроек
        add_action('admin_init', [$this,'register_settings']);

        // логика защиты
        add_filter('comment_form_default_fields', [$this,'inject_timer_field'], 20);
        add_filter('comment_form_fields',        [$this,'inject_timer_field'], 20);
        add_action('pre_comment_on_post',        [$this,'validate_timer_field'], 5);

        // управление HTML5/novalidate (опционально)
        add_filter('comment_form_defaults',      [$this,'maybe_force_xhtml'], 10);

        // активация/удаление
        register_activation_hook(__FILE__, [$this,'on_activate']);
        register_uninstall_hook(__FILE__, ['Comment_Stealth_Timer','on_uninstall']);
    }

    /* ---------------------- Helpers ---------------------- */

    private function is_network_forced(): bool {
        return is_multisite() ? (bool) get_site_option(self::OPT_NET_FORCE, false) : false;
    }
    private function is_enabled_for_site(): bool {
        return $this->is_network_forced() ? true : (bool) get_option(self::OPT_SITE_ENABLED, false);
    }
    private function get_seconds(): int {
        $v = (int) get_option(self::OPT_SITE_SECONDS, self::DEFAULT_SECONDS);
        return max(0, $v);
    }
    private function get_field(): string {
        $v = trim( (string) get_option(self::OPT_SITE_FIELD, self::DEFAULT_FIELD) );
        return $v !== '' ? $v : self::DEFAULT_FIELD;
    }
    private function get_status_code(): int {
        $v = (int) get_option(self::OPT_SITE_STATUS_CODE, self::DEFAULT_STATUS_CODE);
        return ($v >= 400 && $v <= 599) ? $v : self::DEFAULT_STATUS_CODE;
    }
    private function get_message(): string {
        $v = (string) get_option(self::OPT_SITE_MESSAGE, self::DEFAULT_MESSAGE);
        return $v !== '' ? $v : self::DEFAULT_MESSAGE;
    }
    private function bypass_for_moderators(): bool {
        return (bool) get_option(self::OPT_SITE_BYPASS_MODS, true);
    }
    private function force_xhtml_mode(): bool {
        return (bool) get_option(self::OPT_SITE_HTML_MODE, false);
    }

    /* ---------------------- Admin UI ---------------------- */

    public function site_menu() {
        add_submenu_page(
            'options-general.php',
            'Comment Stealth Timer',
            'Comment Stealth Timer',
            'manage_options',
            'comment-stealth-timer',
            [$this,'render_site_page']
        );
    }
    public function network_menu() {
        add_submenu_page(
            'settings.php',
            'Comment Stealth Timer (Network)',
            'Comment Stealth Timer',
            'manage_network_options',
            'comment-stealth-timer-network',
            [$this,'render_network_page']
        );
    }

    public function register_settings() {
        // САЙТОВЫЕ
        register_setting('cst_site_group', self::OPT_SITE_ENABLED, [
            'type'=>'boolean','sanitize_callback'=>function($v){return (bool)$v;}, 'default'=>false
        ]);
        register_setting('cst_site_group', self::OPT_SITE_SECONDS, [
            'type'=>'integer','sanitize_callback'=>function($v){return max(0,intval($v));}, 'default'=>self::DEFAULT_SECONDS
        ]);
        register_setting('cst_site_group', self::OPT_SITE_FIELD, [
            'type'=>'string','sanitize_callback'=>function($v){$v=preg_replace('~[^a-zA-Z0-9_\-]~','',$v); return $v?:self::DEFAULT_FIELD;}, 'default'=>self::DEFAULT_FIELD
        ]);
        register_setting('cst_site_group', self::OPT_SITE_STATUS_CODE, [
            'type'=>'integer','sanitize_callback'=>function($v){$v=intval($v); return ($v>=400 && $v<=599)?$v:self::DEFAULT_STATUS_CODE;}, 'default'=>self::DEFAULT_STATUS_CODE
        ]);
        register_setting('cst_site_group', self::OPT_SITE_MESSAGE, [
            'type'=>'string','sanitize_callback'=>function($v){$v=trim((string)$v); return $v!==''?$v:self::DEFAULT_MESSAGE;}, 'default'=>self::DEFAULT_MESSAGE
        ]);
        register_setting('cst_site_group', self::OPT_SITE_BYPASS_MODS, [
            'type'=>'boolean','sanitize_callback'=>function($v){return (bool)$v;}, 'default'=>true
        ]);
        register_setting('cst_site_group', self::OPT_SITE_HTML_MODE, [
            'type'=>'boolean','sanitize_callback'=>function($v){return (bool)$v;}, 'default'=>false
        ]);

        // СЕТЕВЫЕ
        if ( is_multisite() && is_network_admin() ) {
            register_setting('cst_network_group', self::OPT_NET_FORCE, [
                'type'=>'boolean','sanitize_callback'=>function($v){return (bool)$v;}, 'default'=>false
            ]);
        }
    }

    public function render_site_page() {
        if ( ! current_user_can('manage_options') ) wp_die('Недостаточно прав.');
        ?>
        <div class="wrap">
            <h1>Comment Stealth Timer</h1>
            <?php if ( $this->is_network_forced() ): ?>
                <div class="notice notice-info"><p>В сети включён режим «Принудительно включить». Настройки «Вкл/Выкл» игнорируются.</p></div>
            <?php endif; ?>
            <form method="post" action="options.php">
                <?php settings_fields('cst_site_group'); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Плагин активен</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPT_SITE_ENABLED);?>" value="1"
                                    <?php checked( get_option(self::OPT_SITE_ENABLED, false), true ); ?>
                                    <?php disabled( $this->is_network_forced(), true ); ?>>
                                Включить защиту комментариев на этом сайте
                            </label>
                            <p class="description">Добавляет скрытое поле с таймером и проверяет «время чтения» до отправки комментария.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Минимальное время чтения (сек)</th>
                        <td>
                            <input type="number" min="0" step="1" name="<?php echo esc_attr(self::OPT_SITE_SECONDS);?>" value="<?php echo esc_attr($this->get_seconds());?>">
                            <p class="description">Если комментарий отправлен быстрее — отклоняется с кодом ответа ниже.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Имя скрытого поля</th>
                        <td>
                            <input type="text" name="<?php echo esc_attr(self::OPT_SITE_FIELD);?>" value="<?php echo esc_attr($this->get_field());?>">
                            <p class="description">Только [a–zA–Z0–9_-]. По умолчанию: <code><?php echo esc_html(self::DEFAULT_FIELD);?></code></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">HTTP-код при блокировке</th>
                        <td>
                            <input type="number" min="400" max="599" name="<?php echo esc_attr(self::OPT_SITE_STATUS_CODE);?>" value="<?php echo esc_attr($this->get_status_code());?>">
                            <p class="description">Рекомендуется 503.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Сообщение при блокировке</th>
                        <td>
                            <input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_SITE_MESSAGE);?>" value="<?php echo esc_attr($this->get_message());?>">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Пропуск модераторов</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPT_SITE_BYPASS_MODS);?>" value="1"
                                    <?php checked( get_option(self::OPT_SITE_BYPASS_MODS, true), true ); ?>>
                                Не применять проверку к пользователям с правом «moderate_comments»
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Отключить HTML5/novalidate</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPT_SITE_HTML_MODE);?>" value="1"
                                    <?php checked( get_option(self::OPT_SITE_HTML_MODE, false), true ); ?>>
                                Принудительно выводить форму в режиме XHTML (убирает <code>novalidate</code>)
                            </label>
                            <p class="description">Опционально. Может помочь против «скриптовых» спамеров, но влияет на фронтенд-валидацию.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Сохранить'); ?>
            </form>

            <p style="margin-top:16px;color:#666;">Идея решения взята с сайта <a href="https://reset.name" target="_blank" rel="nofollow">reset.name</a>.</p>
        </div>
        <?php
    }

    public function render_network_page() {
        if ( ! current_user_can('manage_network_options') ) wp_die('Недостаточно прав.');
        ?>
        <div class="wrap">
            <h1>Comment Stealth Timer — настройки сети</h1>
            <form method="post" action="edit.php?action=update">
                <?php wp_nonce_field('cst_network_update','cst_network_nonce'); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Принудительно включить на всех сайтах</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPT_NET_FORCE);?>" value="1"
                                    <?php checked( (bool)get_site_option(self::OPT_NET_FORCE,false), true ); ?>>
                                Игнорировать локальные настройки сайтов и включить везде
                            </label>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Сохранить по сети'); ?>
            </form>
        </div>
        <?php
        // обработка POST
        if ( $_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['cst_network_nonce']) && wp_verify_nonce($_POST['cst_network_nonce'],'cst_network_update') ) {
            update_site_option(self::OPT_NET_FORCE, ! empty($_POST[self::OPT_NET_FORCE]) );
            add_action('network_admin_notices', function(){
                echo '<div class="updated"><p>Настройки сети сохранены.</p></div>';
            });
        }
    }

    /* ---------------------- Core logic ---------------------- */

    public function inject_timer_field( $fields ) {
        if ( ! $this->is_enabled_for_site() ) return $fields;
        if ( is_admin() ) return $fields;

        $field = $this->get_field();

        $html  = '<input type="text" name="'.esc_attr($field).'" required style="display:none" aria-hidden="true">';
        $html .= '<script>(function(){var set=function(){var f=document.getElementsByName("'.esc_js($field).'");if(f&&f[0]){f[0].value=Math.floor(Date.now()/1000).toString();}};if(document.readyState==="loading"){document.addEventListener("DOMContentLoaded",set);}else{set();}})();</script>';

        // в оба набора полей — так мы накроем все вариации разметки
        if ( is_array($fields) ) {
            // comment_form_default_fields — ассоц. массив
            $fields[$field] = ( isset($fields[$field]) ? $fields[$field] : '' ) . "\n" . $html;
            return $fields;
        }
        // comment_form_fields — может быть массив полей (включая textarea) — вставим в конец
        if ( is_array($fields) ) {
            $fields[] = $html;
        }
        return $fields;
    }

    public function validate_timer_field( $comment_post_ID ) {
        if ( ! $this->is_enabled_for_site() ) return;

        // байпас для модераторов/админов (настраиваемо)
        if ( $this->bypass_for_moderators() && is_user_logged_in() && current_user_can('moderate_comments') ) {
            return;
        }

        $field  = $this->get_field();
        $pause  = $this->get_seconds();
        $status = $this->get_status_code();
        $msg    = $this->get_message();

        $val = isset($_POST[$field]) ? $_POST[$field] : '';

        // строгая проверка: 10-значный UNIX timestamp
        if ( ! is_string($val) || ! preg_match('/^\d{10}$/', $val) ) {
            wp_die( esc_html($msg), esc_html($msg), ['response'=>$status] );
        }

        $ts = intval($val);
        if ( $pause > 0 && ( time() - $ts ) < $pause ) {
            wp_die( esc_html($msg), esc_html($msg), ['response'=>$status] );
        }
    }

    public function maybe_force_xhtml( $defaults ) {
        if ( ! $this->is_enabled_for_site() ) return $defaults;
        if ( $this->force_xhtml_mode() ) {
            $defaults['format'] = 'xhtml'; // отключает html5 и novalidate
        }
        return $defaults;
    }

    /* ---------------------- Lifecycle ---------------------- */

    public function on_activate() {
        // значения по умолчанию
        if ( false === get_option(self::OPT_SITE_ENABLED, null) )     add_option(self::OPT_SITE_ENABLED, false);
        if ( false === get_option(self::OPT_SITE_SECONDS, null) )     add_option(self::OPT_SITE_SECONDS, self::DEFAULT_SECONDS);
        if ( false === get_option(self::OPT_SITE_FIELD, null) )       add_option(self::OPT_SITE_FIELD, self::DEFAULT_FIELD);
        if ( false === get_option(self::OPT_SITE_STATUS_CODE, null) ) add_option(self::OPT_SITE_STATUS_CODE, self::DEFAULT_STATUS_CODE);
        if ( false === get_option(self::OPT_SITE_MESSAGE, null) )     add_option(self::OPT_SITE_MESSAGE, self::DEFAULT_MESSAGE);
        if ( false === get_option(self::OPT_SITE_BYPASS_MODS, null) ) add_option(self::OPT_SITE_BYPASS_MODS, true);
        if ( false === get_option(self::OPT_SITE_HTML_MODE, null) )   add_option(self::OPT_SITE_HTML_MODE, false);

        if ( is_multisite() && false === get_site_option(self::OPT_NET_FORCE, null) ) {
            add_site_option(self::OPT_NET_FORCE, false);
        }
    }

    public static function on_uninstall() {
        // Чистим только опции; формы/контент не трогаем
        if ( is_multisite() ) {
            delete_site_option(self::OPT_NET_FORCE);
            $sites = get_sites(['number'=>0]);
            foreach ( $sites as $s ) {
                switch_to_blog( (int) $s->blog_id );
                delete_option(self::OPT_SITE_ENABLED);
                delete_option(self::OPT_SITE_SECONDS);
                delete_option(self::OPT_SITE_FIELD);
                delete_option(self::OPT_SITE_STATUS_CODE);
                delete_option(self::OPT_SITE_MESSAGE);
                delete_option(self::OPT_SITE_BYPASS_MODS);
                delete_option(self::OPT_SITE_HTML_MODE);
                restore_current_blog();
            }
        } else {
            delete_option(self::OPT_SITE_ENABLED);
            delete_option(self::OPT_SITE_SECONDS);
            delete_option(self::OPT_SITE_FIELD);
            delete_option(self::OPT_SITE_STATUS_CODE);
            delete_option(self::OPT_SITE_MESSAGE);
            delete_option(self::OPT_SITE_BYPASS_MODS);
            delete_option(self::OPT_SITE_HTML_MODE);
        }
    }
}

new Comment_Stealth_Timer();
